import { supabase } from '@/config/supabase';
import {
  Admin,
  AdminSettings,
  FeatureFlags,
  SupportTicket,
  AdminLog,
  CreateAdminRequest,
  UpdateAdminPermissionsRequest,
  UpdateSettingsRequest,
  UpdateFeatureFlagsRequest,
  ImpersonateRequest,
  UpdateSupportTicketRequest
} from './types';

export const adminService = {
  // List all admins
  async listAdmins() {
    const { data, error } = await supabase.from('users').select('*').in('role', ['admin', 'super_admin', 'moderator']);
    return { data, error };
  },

  // Create admin or assign role
  async createAdmin(req: CreateAdminRequest) {
    // Check if user exists
    const { data: user } = await supabase.from('users').select('*').eq('email', req.email).single();
    if (user) {
      // Update role/permissions
      const { data, error } = await supabase.from('users').update({ role: req.role, permissions: req.permissions }).eq('id', user.id).select().single();
      return { data, error };
    } else {
      // Invite flow (simulate)
      const { data, error } = await supabase.from('users').insert({ email: req.email, role: req.role, permissions: req.permissions }).select().single();
      // TODO: Send invite email
      return { data, error };
    }
  },

  // Update admin permissions
  async updateAdminPermissions(id: string, req: UpdateAdminPermissionsRequest) {
    const { data, error } = await supabase.from('users').update({ permissions: req.permissions }).eq('id', id).select().single();
    return { data, error };
  },

  // View admin settings
  async getSettings() {
    const { data, error } = await supabase.from('admin_settings').select('*').single();
    return { data, error };
  },

  // Update admin settings
  async updateSettings(req: UpdateSettingsRequest) {
    const { data, error } = await supabase.from('admin_settings').update(req).select().single();
    return { data, error };
  },

  // View admin logs
  async getLogs() {
    const { data, error } = await supabase.from('admin_logs').select('*').order('created_at', { ascending: false });
    return { data, error };
  },

  // Update feature flags
  async updateFeatureFlags(req: UpdateFeatureFlagsRequest) {
    const { data, error } = await supabase.from('feature_flags').update(req).select().single();
    return { data, error };
  },

  // Approve KYC / pricing agent requests
  async approveKYCOrPricing(id: string, status: string, admin_notes: string) {
    // Update kyc_requests or pricing_agents table as needed
    const { data, error } = await supabase.from('kyc_requests').update({ status, admin_notes }).eq('id', id).select().single();
    return { data, error };
  },

  // Impersonate (returns user session token or info)
  async impersonate(admin: any, req: ImpersonateRequest) {
    if (admin.role !== 'super_admin') return { error: { message: 'Only super admin can impersonate.' } };
    // Simulate: fetch user info
    const { data: user, error } = await supabase.from('users').select('*').eq('id', req.user_id).single();
    if (error) return { error };
    // TODO: Generate session token for impersonation
    return { data: user };
  },

  // Audit trail viewer
  async getAuditTrail() {
    const { data, error } = await supabase.from('audit_trail').select('*').order('created_at', { ascending: false });
    return { data, error };
  },

  // Support ticket inbox
  async getSupportTickets() {
    const { data, error } = await supabase.from('support_tickets').select('*').order('created_at', { ascending: false });
    return { data, error };
  },

  // Update support ticket
  async updateSupportTicket(id: string, req: UpdateSupportTicketRequest) {
    const { data, error } = await supabase.from('support_tickets').update(req).eq('id', id).select().single();
    return { data, error };
  },

  // Admin notifications
  async getAdminNotifications(admin_id: string) {
    const { data, error } = await supabase.from('notifications').select('*').eq('user_id', admin_id).order('created_at', { ascending: false });
    return { data, error };
  }
}; 